package gov.va.genisis2.bo;

import java.io.File;
import java.util.List;
import java.util.StringJoiner;

import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.stereotype.Component;

import gov.va.genisis2.common.enums.WorkflowStatusEnum;
import gov.va.genisis2.util.rest.helper.ResponseWrapper;
import gov.va.genisis2.vo.CopyTableSource;

/**
 * The Class BusinessHelper.
 */
@Component
public class BusinessHelper {

	private static final Logger LOGGER = LoggerFactory.getLogger(BusinessHelper.class);

	private static final String EMAILID_DELIMETER = ",";

	@Resource(name="workflowDeadendsList")
	private List<String> workflowDeadendsList;
	
	@Resource(name="allowedOperations")
	private List<String> allowedOperations;
	
	@Resource(name="notificationAllowedStutuses")
	private List<String> notificationAllowedStutuses;
	
	@Resource(name="returnToDdmAllowedStutuses")
	private List<String> returnToDdmAllowedStutuses;

	/**
	 * @param workflowDeadendsList
	 *            the workflowDeadendsList to set
	 */
	@Required
	public void setWorkflowDeadendsList(List<String> workflowDeadendsList) {
		this.workflowDeadendsList = workflowDeadendsList;
	}

	public ResponseWrapper validateCopyTableSource(CopyTableSource copyTableSource, CopyTableSource jsonRequest, ResponseWrapper wrapper) {

		// Get required properties
		String tcSourceDataBaseServerName = copyTableSource.getTcSourceDataBaseServerName();
		String tcSourceDataBaseName = copyTableSource.getTcSourceDataBaseName();
		String sourceSchemaName = jsonRequest.getSourceSchemaName();
		String tcDestinationDataBaseServerName = copyTableSource.getTcDestinationDataBaseServerName();
		String tcDestinationDataBaseUserName = copyTableSource.getTcDestinationDataBaseUserName();
		String tcDestinationDataBaseUserPassword = copyTableSource.getTcDestinationDataBaseUserPassword();
		String tcDestinationServerDataBaseName = copyTableSource.getTcDestinationServerDataBaseName();
		String destinationSchemaName = jsonRequest.getDestinationSchemaName();
		String tcManagementDataBaseServerName = copyTableSource.getTcManagementDataBaseServerName();
		String tcManagementDataBaseUserName = copyTableSource.getTcManagementDataBaseUserName();
		String tcManagementDataBaseUserPassword = copyTableSource.getTcManagementDataBaseUserPassword();
		String tcManagementDataBaseSchema = copyTableSource.getTcManagementDataBaseSchema();
		String copyTable = jsonRequest.getTableName();
		String copyTableScript = copyTableSource.getCmdFilePath();
		String tcManagementDataBaseName = copyTableSource.getTcManagementDataBaseName();
		String tcTemporaryDataBaseName = copyTableSource.getTcTemporaryDataBaseName();

		boolean sourceDBFlag = StringUtils.isBlank(tcSourceDataBaseServerName) || StringUtils.isBlank(tcSourceDataBaseName);
		boolean schemaFlag = StringUtils.isBlank(sourceSchemaName);
		boolean destDBFlag = StringUtils.isBlank(tcDestinationDataBaseServerName) || StringUtils.isBlank(tcDestinationDataBaseUserName) || StringUtils.isBlank(tcDestinationDataBaseUserPassword);
		boolean sourceSchemaDestFlag = sourceDBFlag || schemaFlag || destDBFlag;

		boolean destServFlag = sourceSchemaDestFlag || StringUtils.isBlank(tcDestinationServerDataBaseName) || StringUtils.isBlank(destinationSchemaName);

		boolean mgmtDBFlag = StringUtils.isBlank(tcManagementDataBaseServerName) || StringUtils.isBlank(tcManagementDataBaseSchema) || StringUtils.isBlank(tcManagementDataBaseName);
		boolean mgmtDBUserFlag = StringUtils.isBlank(tcManagementDataBaseUserName) || StringUtils.isBlank(tcManagementDataBaseUserPassword);
		boolean destMgmtflag = destServFlag || mgmtDBFlag || mgmtDBUserFlag;

		boolean copyTableFlag = StringUtils.isBlank(tcTemporaryDataBaseName) || StringUtils.isBlank(copyTable) || StringUtils.isBlank(copyTableScript);

		// Check for all properties are present
		if (sourceSchemaDestFlag || destMgmtflag || copyTableFlag) {
			LOGGER.error("One or more properties in genisis2.properties file seems blank");
			wrapper.setResponse(jsonRequest);
			wrapper.setMessage("One or more properties in genisis2.properties file seems blank");
			wrapper.setSuccess(false);
			return wrapper;
		}

		File file = new File(copyTableScript);

		// Check shell script available on Linux server
		if (!file.isFile()) {
			LOGGER.error("The file {} does not exist", copyTableScript);
			wrapper.setResponse(jsonRequest);
			wrapper.setMessage("Copy table failed. " + copyTableScript + " does not exist");
			wrapper.setSuccess(false);
			return wrapper;
		}

		/*
		 * IP          GENISISDATASOURCE dflt IP          data_ops genisis20!7
		 * GENISISDATADEST dbo IP          data_ops genisis20!7 dbo
		 * vinciTable1,vinciTable2
		 */

		String[] command = new String[16];
		command[0] = copyTableScript;
		command[1] = tcSourceDataBaseServerName;
		command[2] = tcSourceDataBaseName;
		command[3] = sourceSchemaName;
		command[4] = tcDestinationDataBaseServerName;
		command[5] = tcDestinationDataBaseUserName;
		command[6] = tcDestinationDataBaseUserPassword;
		command[7] = tcDestinationServerDataBaseName;
		command[8] = destinationSchemaName;
		command[9] = tcManagementDataBaseServerName;
		command[10] = tcManagementDataBaseUserName;
		command[11] = tcManagementDataBaseUserPassword;
		command[12] = tcManagementDataBaseName;
		command[13] = tcTemporaryDataBaseName;
		command[14] = tcManagementDataBaseSchema;
		command[15] = copyTable;

		wrapper.setResponse(command);
		wrapper.setSuccess(true);

		return wrapper;
	}

	public boolean getWorkflowDeadEnds(String status) {
		boolean workflowDeadEnds = false;
		StringJoiner terminatedStates = new StringJoiner(EMAILID_DELIMETER);
		for (Object obj : workflowDeadendsList) {
			if (obj != null) {
				terminatedStates.add(obj.toString());
			}
		}
		if (StringUtils.isBlank(terminatedStates.toString())) {
			LOGGER.error("No workflow deadends found!");
			workflowDeadEnds = true;
		}
		if (StringUtils.countMatches(terminatedStates.toString(), status) > 0) {
			workflowDeadEnds = true;
		}
		return workflowDeadEnds;
	}
	
	public boolean isCancelAllowed(String status) {
		boolean flag;
		flag = StringUtils.equalsIgnoreCase(status, WorkflowStatusEnum.SUBMITTED.getDesc()) ? true : false;
		flag = flag || (StringUtils.equalsIgnoreCase(status, WorkflowStatusEnum.RETURNED.getDesc())) ? true : false;
		flag = flag || (StringUtils.equalsIgnoreCase(status, WorkflowStatusEnum.SENT.getDesc())) ? true : false;
		flag = flag || (StringUtils.equalsIgnoreCase(status, WorkflowStatusEnum.REQUESTACCEPTED.getDesc())) ? true : false;
		flag = flag || (StringUtils.equalsIgnoreCase(status, WorkflowStatusEnum.RESULTSDELIVERED.getDesc())) ? true : false;
		return flag;
	}

	public boolean isOperationAllowed(String operation) {
		return allowedOperations.stream().filter(s -> s.equalsIgnoreCase(operation)).findFirst().isPresent();
	}
	
	public boolean isNotificationAllowed(String status) {
		return notificationAllowedStutuses.stream().filter(s -> s.equalsIgnoreCase(status)).findFirst().isPresent();
	}
	
	public boolean isReturnToDDMAllowed(String status) {
		return returnToDdmAllowedStutuses.stream().filter(s -> s.equalsIgnoreCase(status)).findFirst().isPresent();
	}
}
